<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use MercadoPago\SDK;
use App\Mail\RegistroUsuarioPlanSiempreJuntos;

class MercadoPagoWebhookController extends Controller
{
    public function handle(Request $request)
    {
        Log::info('📥 Webhook recibido', [
            'body' => $request->all()
        ]);

        $type = $request->input('type');
        $dataId = $request->input('data.id');

        if ($type !== 'payment' || !$dataId) {
            Log::info('ℹ️ Evento ignorado', ['type' => $type]);
            return response()->json(['ignored' => true]);
        }

        try {

            // 🔐 Token Mercado Pago
            $accessToken = config('services.mercadopago.access_token');

            if (!$accessToken) {
                Log::error('❌ Access token no configurado');
                return response()->json(['error' => 'token_missing'], 500);
            }

            SDK::setAccessToken($accessToken);

            // 💳 Obtener pago
            $response = Http::withToken($accessToken)
                ->get("https://api.mercadopago.com/v1/payments/{$dataId}");

            if ($response->failed()) {
                Log::error('❌ No se pudo obtener el pago', [
                    'dataId' => $dataId,
                    'response' => $response->body()
                ]);
                return response()->json(['error' => 'payment_not_found'], 400);
            }

            $payment = $response->json();

            Log::info('💳 Detalle del pago', [
                'id' => $payment['id'],
                'status' => $payment['status']
            ]);

            // ✔ SOLO pagos aprobados
            if (($payment['status'] ?? null) !== 'approved') {
                return response()->json(['ignored' => 'payment_not_approved']);
            }

            // 🔗 external_reference
            $refData = json_decode($payment['external_reference'] ?? '', true);

            // Si viene de suscripción, reforzamos desde preapproval
            if (isset($payment['preapproval_id'])) {

                $preapprovalResponse = Http::withToken($accessToken)
                    ->get("https://api.mercadopago.com/preapproval/{$payment['preapproval_id']}");

                if ($preapprovalResponse->ok()) {
                    $preapproval = $preapprovalResponse->json();
                    $refData = json_decode($preapproval['external_reference'], true) ?? $refData;
                }
            }

            if (!$refData || !isset($refData['pedido_id'], $refData['pagos_id'])) {
                Log::error('❌ external_reference inválido', ['ref' => $refData]);
                return response()->json(['error' => 'invalid_reference'], 400);
            }

            // 🔒 Evitar duplicar cuotas
            $yaExisteCuota = DB::table('pagos_cuotas')
                ->where('payment_mp_id', $payment['id'])
                ->exists();

            if ($yaExisteCuota) {
                Log::warning('⚠️ Pago ya registrado, webhook repetido');
                return response()->json(['duplicated' => true]);
            }

            DB::transaction(function () use ($payment, $refData) {

                // 1️⃣ Registrar cuota
                DB::table('pagos_cuotas')->insert([
                    'id_pago' => $refData['pagos_id'],
                    'payment_mp_id' => $payment['id'],
                    'numero_cuota' => 1,
                    'fecha_programada' => now(),
                    'fecha_pagada' => now(),
                    'monto' => $payment['transaction_amount'],
                    'estado' => 'pagado',
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                // 2️⃣ Marcar pago principal
                DB::table('pagos')
                    ->where('id_cuota_pago', $refData['pagos_id'])
                    ->update([
                        'estado' => 'pagado',
                        'updated_at' => now()
                    ]);

                // 3️⃣ Activar pedido
                DB::table('pedidos')
                    ->where('id', $refData['pedido_id'])
                    ->update([
                        'estado_pedido' => 1,
                        'pedido_activo' => 1
                    ]);
            });

            // 4️⃣ Enviar correo SOLO UNA VEZ
            $pago = DB::table('pagos')
                ->where('id_cuota_pago', $refData['pagos_id'])
                ->first();


            $pedido = DB::table('pedidos')->where('id_pedido', $refData['pedido_id'])->first();
            $usuario = DB::table('usuarios')->where('id_usuario', $pedido->id_usuario)->first();
            $argsMascota = DB::table('mascotas')->where('id_mascota', $pedido->id_mascota)->first();
            $argRecojo = DB::table('recojos')->where('id_recojo_ida', $pedido->id_recojo_ida)->first();


            $argsMailUsuarioNuevo = [
                'nombre_usuario' => $usuario->nombres . ' ' . $usuario->apellidos,
                'celular' => $usuario->celular,
                'nombre_mascota' => $argsMascota->nombre_mascota,
                'raza_mascota' => $argsMascota->raza_mascota,
                'fecha_recojo' => $argRecojo->fecha_recojo,
                'id_horario' => $argRecojo->hora_recojo,
                'direccion_recojo' => $argRecojo->direccion_recojo,
                'forma_pago' => 'Mercado Pago',
            ];

            if ($usuario->estado == 0) {

                Mail::to(config('mail.admin_address'))->send(new RegistroUsuarioNuevo($argsMailUsuarioNuevo));
                Mail::to($usuario->email)
                    ->send(new RegistroUsuarioPlanSiempreJuntos([
//                        'nombre_usuario' => $usuario->nombres
                        'nombre_usuario' => $usuario->nombres . ' ' . $usuario->apellidos,
                        'nombre_mascota' => $argsMascota->nombre_mascota,
                        'token_reset' => $usuario->token_reset
                    ]));

                DB::table('usuarios')
                    ->where('id', $pedido->id_usuario)
                    ->update([
                        'estado' => 1,
                    ]);
            }


            Log::info('✅ Webhook procesado correctamente y correos enviados', [
                'payment_id' => $payment['id']
            ]);

            return response()->json(['success' => true]);

        } catch (\Throwable $e) {

            Log::error('❌ Error crítico en webhook', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json(['error' => 'server_error'], 500);
        }
    }
}
